// Adds a prefix to the titles of HTML pages.
//

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <strings.h>
#include "nfapi_linux.h"
#include "ProtocolFilters.h"
#include "PFEventsDefault.h"

using namespace nfapi;
using namespace ProtocolFilters;

// Change this string after renaming and registering the driver under different name
#define NFDRIVER_NAME "netfilter2"

std::string g_titlePrefix;

class HttpFilter : public PFEventsDefault
{
public:
	HttpFilter()
	{
	}

	virtual void tcpConnected(nfapi::ENDPOINT_ID id, nfapi::PNF_TCP_CONN_INFO pConnInfo)
	{
		if (pConnInfo->direction == NF_D_OUT)
		{
			pf_addFilter(id, FT_PROXY, FF_READ_ONLY_IN | FF_READ_ONLY_OUT);
			pf_addFilter(id, FT_SSL, FF_SSL_INDICATE_HANDSHAKE_REQUESTS); 
			pf_addFilter(id, FT_HTTP, FF_HTTP_BLOCK_SPDY);
			pf_addFilter(id, FT_HTTP2);
		}
	}

	bool updateContent(PFObject * object)
	{
		PFStream * pStream = object->getStream(HS_CONTENT);
		char * buf;
		bool contentUpdated = false;

		if (pStream && pStream->size() > 0)
		{
			buf = (char*)malloc((size_t)pStream->size() + 1);
			if (buf)
			{
				pStream->read(buf, (tStreamSize)pStream->size());
				buf[pStream->size()] = '\0';

				for (char * p = buf; *p; p++)
				{
					if (strncasecmp(p, "<title", 6) == 0)
					{
						p = p + 6;
						if (p = strchr(p, '>'))
						{
							tStreamSize len = (tStreamSize)pStream->size();
							pStream->reset();
							pStream->write(buf, (tStreamSize)(p - buf + 1));
							pStream->write(g_titlePrefix.c_str(), (tStreamSize)g_titlePrefix.length());
							pStream->write(p + 1, len - (p + 1 - buf));
							
							contentUpdated = true;
						}
						break;
					}
				}
				free(buf);
			}
		}

		return contentUpdated;
	}

	void dataAvailable(nfapi::ENDPOINT_ID id, PFObject * object)
	{
		if (object->isReadOnly())
			return;

		if ((object->getType() == OT_HTTP_RESPONSE) ||
			(object->getType() == OT_HTTP2_RESPONSE))
		{
			PFHeader h;

			if (pf_readHeader(object->getStream(HS_HEADER), &h))
			{
				PFHeaderField * pField = h.findFirstField("Content-Type");
				if (pField)
				{
					if (pField->value().find("text/html") == -1)
					{
						pf_postObject(id, object);
						return;
					}
				}
			}

			updateContent(object);
		}

		pf_postObject(id, object);
	}

	PF_DATA_PART_CHECK_RESULT 
	dataPartAvailable(nfapi::ENDPOINT_ID id, PFObject * object)
	{
		if (object->getType() == OT_SSL_HANDSHAKE_OUTGOING)
		{
			PFStream * pStream = object->getStream(0);
			char * buf;
			PF_DATA_PART_CHECK_RESULT res = DPCR_FILTER;

			if (pStream && pStream->size() > 0)
			{
				buf = (char*)malloc((size_t)pStream->size() + 1);
				if (buf)
				{
					pStream->read(buf, (tStreamSize)pStream->size());
					buf[pStream->size()] = '\0';

					if (strcmp(buf, "get.adobe.com") == 0)
					{
						res = DPCR_BYPASS;
					}

					free(buf);
				}
			}
			return res;
		}

		if (object->getType() == OT_HTTP_RESPONSE ||
			object->getType() == OT_HTTP2_RESPONSE)
		{
			PFHeader h;

			if (pf_readHeader(object->getStream(HS_HEADER), &h))
			{
				PFHeaderField * pField = h.findFirstField("Content-Type");
				if (pField)
				{
					if (pField->value().find("text/html") != -1)
					{
						if (updateContent(object))
						{
							return DPCR_UPDATE_AND_BYPASS;
						} else
						{
							return DPCR_MORE_DATA_REQUIRED;
						}
					}
				}
			}
		}

		return DPCR_BYPASS;
	}
};

int main(int argc, char* argv[])
{
	NF_RULE rule;

	if (argc < 2)
	{
		printf("Usage: PFHttpContentFilter <string>\n" \
			"<string> : add this to titles of HTML pages\n");
		return -1;
	}
	
	g_titlePrefix = argv[1];
	g_titlePrefix += " ";
	
#ifdef _DEBUG
	_CrtSetDbgFlag(_CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF);
	_CrtSetReportMode( _CRT_ERROR, _CRTDBG_MODE_DEBUG);
#endif

//	nf_adjustProcessPriviledges();

#ifdef WIN32
#warning WIN32 define on LINUX
#endif
//	nf_setOptions(0, 0);

	printf("Press any key to stop...\n\n");

	HttpFilter f;

	if (!pf_init(&f, "netfilter2"))
	{
		printf("Failed to initialize protocol filter");
		return -1;
	}

	pf_setExceptionsTimeout(EXC_GENERIC, 30);
	pf_setExceptionsTimeout(EXC_TLS, 30);
	pf_setExceptionsTimeout(EXC_CERT_REVOKED, 30);

	pf_setRootSSLCertSubject("Sample CA");

	if (!pf_loadCAStore("/etc/ssl/certs/ca-bundle.crt"))
	{
		if (!pf_loadCAStore("/etc/ssl/certs/ca-certificates.crt"))
		{
			if (!pf_loadCAStore("/etc/pki/tls/certs/ca-bundle.crt"))
			{
				printf("Failed to load root certificates\n");
			}
		}
	}

	// Initialize the library and start filtering thread
	if (nf_init(NFDRIVER_NAME, pf_getNFEventHandler()) != NF_STATUS_SUCCESS)
	{
		printf("Failed to connect to driver");
		return -1;
	}


	NFEXT_RULE rules[3];
	PNFEXT_RULE pRule;
	
	// Block QUIC

	pRule = &rules[0];

	memset(pRule, 0, sizeof(NFEXT_RULE));
	pRule->ruleType = NFEXT_PACKET_RULE;
	pRule->protocol = IPPROTO_UDP;
	pRule->fieldsMask = NFEXT_USE_DST_PORTS;
	pRule->dstPorts.from = pRule->dstPorts.to = 443;
	pRule->filteringFlag = NFEXT_BLOCK;

	pRule = &rules[1];

	memset(pRule, 0, sizeof(NFEXT_RULE));
	pRule->ruleType = NFEXT_PACKET_RULE;
	pRule->protocol = IPPROTO_UDP;
	pRule->fieldsMask = NFEXT_USE_DST_PORTS;
	pRule->dstPorts.from = pRule->dstPorts.to = 80;
	pRule->filteringFlag = NFEXT_BLOCK;

	// Filter TCP

	pRule = &rules[2];

	memset(pRule, 0, sizeof(NFEXT_RULE));
	pRule->ruleType = NFEXT_REDIRECT_RULE;
	pRule->filteringFlag = NFEXT_REDIRECT;

	if (nf_setRules(rules, 3) != NF_STATUS_SUCCESS)
	{
		printf("Failed to add filtering rules. Root rights are required.\n");
		return -1;
	}

	// Wait for any key
	getchar();

	// Free the libraries
	nf_free();
	pf_free();

	return 0;
}

