//
// 	NetFilterSDK 
// 	Copyright (C) Vitaly Sidorov
//	All rights reserved.
//
//	This file is a part of the NetFilter SDK.
//	The code and information is provided "as-is" without
//	warranty of any kind, either expressed or implied.
//

#include "vdefs.h"
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/un.h>
#include <sys/errno.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdarg.h>
#include <net/if.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "dbglog.h"
#include "nfproxy.h"
#include "nfapi_linux.h"

#include "vbuffer.h"
#include "vlist.h"

using namespace mainNS;

#define NF_MAX_NAME 512

struct NFRULES_DATA
{
	char driverName[NF_MAX_NAME];

	char driverName_redirect_out[NF_MAX_NAME];
	char driverName_redirect_in[NF_MAX_NAME];

	char driverName_in[NF_MAX_NAME];
	char driverName_out[NF_MAX_NAME];
};

static NFRULES_DATA g_nfRulesData;

#define NF_MAX_PARAMS 100

struct PARAM_LIST
{
	VLIST_ENTRY 	entry;
	unsigned int nParams;
	char * params[NF_MAX_PARAMS];
};

#define FWCTRL_PROCESS_IP "ip"
#define FWCTRL_PROCESS_IPV4 "iptables"
#define FWCTRL_PROCESS_IPV6 "ip6tables"
#define FWCTRL_FILTER "filter"
#define FWCTRL_INPUT "INPUT"
#define FWCTRL_OUTPUT "OUTPUT"
#define FWCTRL_PREROUTING "PREROUTING"
#define FWCTRL_POSTROUTING "POSTROUTING"
#define FWCTRL_MANGLE "mangle"
#define FWCTRL_NAT "nat"

#if defined(_DEBUG) || defined(_RELEASE_LOG)
#define DbgPrintParamList(p) _DbgPrintParamList(p)

void _DbgPrintParamList(const PARAM_LIST * params)
{
	vbuffer_t s;

	s = vbuffer_create();
	if (!s)
		return;

	for (int i=0; i<params->nParams; i++)
	{
		if (!vbuffer_appendString(s, params->params[i]))
			break;

		if (!vbuffer_appendString(s, " "))
			break;
	}

	DbgPrint("%s", vbuffer_buffer(s));

	vbuffer_free(s);
}

#else
#define DbgPrintParamList(p)
#endif

static PARAM_LIST * nfrules_paramListCreate()
{
	PARAM_LIST * p;

	p = vmalloc(PARAM_LIST);
	if (!p)
		return NULL;

	memset(p, 0, sizeof(PARAM_LIST));

	return p;
}

static void nfrules_paramListFree(PARAM_LIST * params)
{
	if (params->nParams > 0)
	{
		for (int i=0; i<params->nParams; i++)
		{
			if (params->params[i])
				free(params->params[i]);
		}
	}

	free(params);
}

static bool nfrules_paramListAddParam(PARAM_LIST * params, const char * param)
{
	if (params->nParams >= NF_MAX_PARAMS)
	{
		DbgPrint("nfrules_addParam too many parameters");
		return false;
	}

	if (param)
	{
		params->params[params->nParams] = strdup(param);
		if (!params->params[params->nParams])
			return false;
	}

	params->nParams++;

	return true;
}

static void nfrules_paramListListFree(VLIST_ENTRY * list)
{
	VLIST_ENTRY * entry;
	PARAM_LIST * params;

	while (!vlist_isEmpty(list))
	{
		entry = vlist_removeHead(list);

		params = vlist_record(entry, PARAM_LIST, entry);

		nfrules_paramListFree(params);
	}
}

static int do_exec(char *const args[])
{
	pid_t pid;
	int status;
	int eo;

	pid = fork();
	if (pid == -1) 
	{
		DbgPrint("fork failed: %s", strerror(errno));
		return -1;
	}
	if (!pid) 
	{
		int fd = open("/dev/null", O_WRONLY);
        if (fd > 0)
		{
			if (dup2(fd, 2)  != 1 )
			{
               //handle error 
			}
        }

		/* In child process, execute external command. */
		(void)execvp(args[0], args);
		/* We only get here if the execv() failed. */
		eo = errno;
//		DbgPrint("execvp %s failed: %s", args[0], strerror(eo));
		_exit(eo);
	}
	/* In parent process, wait for exernal command to finish. */
	if (wait4(pid, (int*)&status, 0, NULL) != pid) 
	{
		DbgPrint("BUG executing %s command.", args[0]);
		return -1;
	}
	if (!WIFEXITED(status)) 
	{
		DbgPrint("%s command aborted by signal %d.",
				args[0], WTERMSIG(status));
		return -1;
	}
	eo = WEXITSTATUS(status);
	if (eo) 
	{
		DbgPrint("%s command failed: %s", 
				args[0], strerror(eo));
		return -1;
	}
	return 0;
}

static bool execParams(const PARAM_LIST * params)
{
	DbgPrint("execParams()");
	DbgPrintParamList(params);

	return do_exec((char*const*)params->params) == 0;
}

static PARAM_LIST * createParamList(const char * processName, ...)
{
	PARAM_LIST * params;
	va_list args;
	char * p;
	
	params = nfrules_paramListCreate();
	if (!params)
		return NULL;

	nfrules_paramListAddParam(params, processName);

	// Add -w parameter to make iptables wait for exclusive lock
	nfrules_paramListAddParam(params, "-w");

	va_start(args, processName);

	p = va_arg(args, char*);
	while (p != NULL)
	{
		nfrules_paramListAddParam(params, p);
		p = va_arg(args, char*);
	}

	va_end(args);

	return params;
}

static PARAM_LIST * createParamListGeneric(const char * processName, ...)
{
	PARAM_LIST * params;
	va_list args;
	char * p;
	
	params = nfrules_paramListCreate();
	if (!params)
		return NULL;

	nfrules_paramListAddParam(params, processName);

	va_start(args, processName);

	p = va_arg(args, char*);
	while (p != NULL)
	{
		nfrules_paramListAddParam(params, p);
		p = va_arg(args, char*);
	}

	va_end(args);

	return params;
}

static vbuffer_t subNetToString(int ipFamily, const char * ipAddr, unsigned char ipAddrMask)
{
	char buf[100];
	vbuffer_t sAddr;

	sAddr = vbuffer_create();
	if (!sAddr)
		return NULL;

	if (ipFamily == AF_INET)
	{
		if (inet_ntop(AF_INET, ipAddr, buf, sizeof(buf)))
		{
			vbuffer_appendString(sAddr, buf);
		}
	} else
	{
		if (inet_ntop(AF_INET6, ipAddr, buf, sizeof(buf)))
		{
			vbuffer_appendString(sAddr, buf);
		}
	}

	if (ipAddrMask != 0)
	{
		vbuffer_appendString(sAddr, "/");
		_snprintf(buf, sizeof(buf), "%d", ipAddrMask);
		vbuffer_appendString(sAddr, buf);
	}

	return sAddr;
}

static vbuffer_t portRangeToString(PNFEXT_PORT_RANGE pPorts)
{
	char buf[100];
	vbuffer_t s;

	s = vbuffer_create();
	if (!s)
		return NULL;

	if (pPorts->from == pPorts->to)
	{
		_snprintf(buf, sizeof(buf), "%d", pPorts->from);
		vbuffer_appendString(s, buf);
		return s;
	} else
	{
		_snprintf(buf, sizeof(buf), "%d:%d", pPorts->from, pPorts->to);
		vbuffer_appendString(s, buf);
		return s;
	}
}

static vbuffer_t intToString(int n)
{
	char buf[100];
	vbuffer_t s;

	s = vbuffer_create();
	if (!s)
		return NULL;

	_snprintf(buf, sizeof(buf), "%d", n);

	vbuffer_appendString(s, buf);

	return s;
}

static PARAM_LIST * createRuleParamList(PNFEXT_RULE pRule,
								int ip_family,
								const char * ctrlProcess,
								const char * table,
								const char * chain)
{
	PARAM_LIST * params;

	DbgPrint("createRuleParamLists()");

	params = createParamList(
					ctrlProcess,
					"-t", table,
					"-A", chain,
					(char*)NULL);
	if (!params)
		return NULL;

	switch (pRule->protocol)
	{
	case 0:
		break;
	case IPPROTO_TCP:
		{
			nfrules_paramListAddParam(params, "-p");
			nfrules_paramListAddParam(params, "tcp");
		}
		break;
	case IPPROTO_UDP:
		{
			nfrules_paramListAddParam(params, "-p");
			nfrules_paramListAddParam(params, "udp");
		}
		break;
	default:
		{
			nfrules_paramListAddParam(params, "-p");
			vbuffer_t s = intToString(pRule->protocol);
			if (s)
			{
				nfrules_paramListAddParam(params, vbuffer_buffer(s));
				vbuffer_free(s);
			}
		}
		break;
	}

	switch (pRule->filteringFlag)
	{
	case NFEXT_BYPASS:
		{
			if (pRule->ruleType == NFEXT_REDIRECT_RULE)
			{
				nfrules_paramListAddParam(params, "-j");
				nfrules_paramListAddParam(params, "RETURN");
			} else
			{
				nfrules_paramListAddParam(params, "-j");
				nfrules_paramListAddParam(params, "ACCEPT");
			}
		}
		break;
	
	case NFEXT_BLOCK:
		{
			if (pRule->ruleType == NFEXT_REDIRECT_RULE)
			{
				if (pRule->protocol == 0)
				{
					nfrules_paramListAddParam(params, "-p");
					nfrules_paramListAddParam(params, "tcp");
				}
			}

			nfrules_paramListAddParam(params, "-j");
			nfrules_paramListAddParam(params, "DROP");
		}
		break;
	
	case NFEXT_REDIRECT:
		{
			if (pRule->ruleType != NFEXT_REDIRECT_RULE)
			{
				DbgPrint("createRuleParamLists() invalid rule type %d", pRule->ruleType);
				nfrules_paramListFree(params);
				return NULL;
			}

			if (pRule->protocol == 0)
			{
				nfrules_paramListAddParam(params, "-p");
				nfrules_paramListAddParam(params, "tcp");
			}

			nfrules_paramListAddParam(params, "-m");
			nfrules_paramListAddParam(params, "mark");
//			params.push_back("!");
			nfrules_paramListAddParam(params, "--mark");
			nfrules_paramListAddParam(params, "0/" MARK_PASS_S);
			nfrules_paramListAddParam(params, "-m");
			nfrules_paramListAddParam(params, "conntrack");
			nfrules_paramListAddParam(params, "--ctstate");
			nfrules_paramListAddParam(params, "NEW,ESTABLISHED");
			nfrules_paramListAddParam(params, "--ctdir");
			nfrules_paramListAddParam(params, "ORIGINAL");
			nfrules_paramListAddParam(params, "-j");
			nfrules_paramListAddParam(params, "MARK");
			nfrules_paramListAddParam(params, "--set-xmark");
			nfrules_paramListAddParam(params, MARK_IN_S"/0xffffffff");
		}
		break;

	default:
		DbgPrint("createRuleParamLists() invalid filtering flag %d", pRule->filteringFlag);
		nfrules_paramListFree(params);
		return NULL;
	}

	if (pRule->fieldsMask & NFEXT_USE_DST_IP)
	{
		nfrules_paramListAddParam(params, "--dst");
		
		vbuffer_t s = subNetToString(ip_family, pRule->dstIp, pRule->dstIpMask);
		if (s)
		{
			nfrules_paramListAddParam(params, vbuffer_buffer(s));
			vbuffer_free(s);
		}
	}

	if (pRule->fieldsMask & NFEXT_USE_SRC_IP)
	{
		nfrules_paramListAddParam(params, "--src");
		
		vbuffer_t s = subNetToString(ip_family, pRule->srcIp, pRule->srcIpMask);
		if (s)
		{
			nfrules_paramListAddParam(params, vbuffer_buffer(s));
			vbuffer_free(s);
		}
	}

	if (pRule->fieldsMask & NFEXT_USE_DST_PORTS)
	{
		nfrules_paramListAddParam(params, "--dport");
		vbuffer_t s = portRangeToString(&pRule->dstPorts);
		if (s)
		{
			nfrules_paramListAddParam(params, vbuffer_buffer(s));
			vbuffer_free(s);
		}
	}

	if (pRule->fieldsMask & NFEXT_USE_SRC_PORTS)
	{
		nfrules_paramListAddParam(params, "--sport");
		vbuffer_t s = portRangeToString(&pRule->srcPorts);
		if (s)
		{
			nfrules_paramListAddParam(params, vbuffer_buffer(s));
			vbuffer_free(s);
		}
	}

	if (pRule->fieldsMask & (NFEXT_USE_UID | NFEXT_USE_GID))
	{
		nfrules_paramListAddParam(params, "-m");
		nfrules_paramListAddParam(params, "owner");
	}

	if (pRule->fieldsMask & NFEXT_USE_UID)
	{
		nfrules_paramListAddParam(params, "--uid-owner");
		vbuffer_t s = intToString(pRule->uid);
		if (s)
		{
			nfrules_paramListAddParam(params, vbuffer_buffer(s));
			vbuffer_free(s);
		}
	}

	if (pRule->fieldsMask & NFEXT_USE_GID)
	{
		nfrules_paramListAddParam(params, "--gid-owner");
		vbuffer_t s = intToString(pRule->gid);
		if (s)
		{
			nfrules_paramListAddParam(params, vbuffer_buffer(s));
			vbuffer_free(s);
		}
	}

	return params;
}


static bool createRuleParamLists(PNFEXT_RULE pRule,
								int ip_family,
								VLIST_ENTRY * list)
{
	PARAM_LIST * params;

	if (pRule->ip_family != 0 &&
		pRule->ip_family != ip_family)
	{
		return true;
	}

	const char * ctrlProcess = 
		(ip_family == AF_INET)? 
		FWCTRL_PROCESS_IPV4 : FWCTRL_PROCESS_IPV6;

	if (pRule->ruleType == NFEXT_REDIRECT_RULE)
	{
#ifdef NO_IPV6_NAT
		if (ip_family == AF_INET6)
		{
			return true;
		}
#endif
		if (pRule->direction == NF_D_IN || pRule->direction == NF_D_BOTH || pRule->direction == 0)
		{
			params = createRuleParamList(pRule, ip_family, ctrlProcess, FWCTRL_MANGLE, g_nfRulesData.driverName_redirect_in);
			if (!params)
			{
				return false;
			}
			vlist_insertTail(list, &params->entry);
		}

		if (pRule->direction == NF_D_OUT || pRule->direction == NF_D_BOTH || pRule->direction == 0)
		{
			params = createRuleParamList(pRule, ip_family, ctrlProcess, FWCTRL_MANGLE, g_nfRulesData.driverName_redirect_out);
			if (!params)
			{
				return false;
			}
			vlist_insertTail(list, &params->entry);
		}
	} else
	if (pRule->ruleType == NFEXT_PACKET_RULE)
	{
		if (pRule->direction == NF_D_IN || pRule->direction == NF_D_BOTH || pRule->direction == 0)
		{
			params = createRuleParamList(pRule, ip_family, ctrlProcess, FWCTRL_FILTER, g_nfRulesData.driverName_in);
			if (!params)
			{
				return false;
			}
			vlist_insertTail(list, &params->entry);
		}

		if (pRule->direction == NF_D_OUT || pRule->direction == NF_D_BOTH || pRule->direction == 0)
		{
			params = createRuleParamList(pRule, ip_family, ctrlProcess, FWCTRL_FILTER, g_nfRulesData.driverName_out);
			if (!params)
			{
				return false;
			}
			vlist_insertTail(list, &params->entry);
		}
	} else
	{
		DbgPrint("createRuleParamLists() failed, invalid rule type %d", pRule->ruleType);
		return false;
	}

	return true;
}

static bool execRulesList(VLIST_ENTRY * list, bool stopOnError = true)
{
	VLIST_ENTRY * entry;
	PARAM_LIST * params;

	vlist_forEach(list, entry)
	{
		params = vlist_record(entry, PARAM_LIST, entry);
		if (!execParams(params))
		{
			DbgPrint("execRulesList() failed");
			
			if (stopOnError)
				return false;
		}
	}

	return true;
}

NFAPI_API NF_STATUS NFAPI_NS 
nf_setRules(PNFEXT_RULE pRules, int count)
{
	VLIST_ENTRY ipv4RulesList;
	VLIST_ENTRY ipv6RulesList;
	char sUid[100];
	char sPort[100];
	NF_STATUS status = NF_STATUS_SUCCESS;

	vlist_init(&ipv4RulesList);
	vlist_init(&ipv6RulesList);

	nf_deleteRules();

	_snprintf(sUid, sizeof(sUid), "%u", geteuid());
	_snprintf(sPort, sizeof(sPort), "%u", nf_getProxyPort());

	DbgPrint("nf_setRules");

	for (;;)
	{
		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_MANGLE,
					"-N", g_nfRulesData.driverName_redirect_out,
					(char*)NULL)
			);

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_MANGLE,
					"-N", g_nfRulesData.driverName_redirect_in,
					(char*)NULL)
			);

	#ifndef NO_IPV6_NAT

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_MANGLE,
					"-N", g_nfRulesData.driverName_redirect_out,
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_MANGLE,
					"-N", g_nfRulesData.driverName_redirect_in,
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_MANGLE,
					"-A", g_nfRulesData.driverName_redirect_out,
					"-p", "tcp",
					"-m", "owner",
					"--uid-owner", sUid,
					"-j", "RETURN",
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_MANGLE,
					"-A", g_nfRulesData.driverName_redirect_out,
					"-p", "tcp",
					"--src", "::ffff:127.0.0.1",
					"-j", "RETURN",
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_MANGLE,
					"-A", g_nfRulesData.driverName_redirect_in,
					"-p", "tcp",
					"--src", "::ffff:127.0.0.1",
					"-j", "RETURN",
					(char*)NULL));
	#endif
		
		// Bypass Docker traffic

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_MANGLE,
					"-A", g_nfRulesData.driverName_redirect_in,
					"-i", "br-+",
					"-j", "RETURN",
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_MANGLE,
					"-A", g_nfRulesData.driverName_redirect_in,
					"-i", "br-+",
					"-j", "RETURN",
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_MANGLE,
					"-A", g_nfRulesData.driverName_redirect_in,
					"-i", "docker0",
					"-j", "RETURN",
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_MANGLE,
					"-A", g_nfRulesData.driverName_redirect_in,
					"-i", "docker0",
					"-j", "RETURN",
					(char*)NULL));

		// Bypass LXC traffic

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_MANGLE,
					"-A", g_nfRulesData.driverName_redirect_in,
					"-i", "lxcbr0",
					"-j", "RETURN",
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_MANGLE,
					"-A", g_nfRulesData.driverName_redirect_in,
					"-i", "lxcbr0",
					"-j", "RETURN",
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_FILTER,
					"-N", g_nfRulesData.driverName_in,
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_FILTER,
					"-N", g_nfRulesData.driverName_out,
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_MANGLE,
					"-A", g_nfRulesData.driverName_redirect_out,
					"-p", "tcp",
					"-m", "owner",
					"--uid-owner", sUid,
					"-j", "RETURN",
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_MANGLE,
					"-A", g_nfRulesData.driverName_redirect_out,
					"-p", "tcp",
					"--src", "127.0.0.1",
					"-j", "RETURN",
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_MANGLE,
					"-A", g_nfRulesData.driverName_redirect_in,
					"-p", "tcp",
					"--src", "127.0.0.1",
					"-j", "RETURN",
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_FILTER,
					"-N", g_nfRulesData.driverName_in,
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_FILTER,
					"-N", g_nfRulesData.driverName_out,
					(char*)NULL));

		{
			NFEXT_RULE rule;
		
			memset(&rule, 0, sizeof(NFEXT_RULE));
			rule.ruleType = NFEXT_PACKET_RULE;
			rule.protocol = IPPROTO_TCP;
			rule.direction = NF_D_IN;
			rule.fieldsMask = NFEXT_USE_DST_PORTS;
			rule.dstPorts.from = rule.dstPorts.to = nf_getProxyPort();
			rule.filteringFlag = NFEXT_BYPASS;

			if (!createRuleParamLists(&rule, AF_INET, &ipv4RulesList))
			{
				DbgPrint("nf_setRules() unable to add allowing rule for proxy");
				status = NF_STATUS_FAIL;
				break;
			}
		}

		{
			NFEXT_RULE rule;
		
			memset(&rule, 0, sizeof(NFEXT_RULE));
			rule.ruleType = NFEXT_PACKET_RULE;
			rule.protocol = IPPROTO_TCP;
			rule.direction = NF_D_IN;
			rule.fieldsMask = NFEXT_USE_DST_PORTS;
			rule.dstPorts.from = rule.dstPorts.to = nf_getProxyPort();
			rule.filteringFlag = NFEXT_BYPASS;

			if (!createRuleParamLists(&rule, AF_INET6, &ipv6RulesList))
			{
				DbgPrint("nf_setRules() unable to add allowing rule for proxy (IPv6)");
				status = NF_STATUS_FAIL;
				break;
			}
		}

		for (int i=0; i<count; i++)
		{
			if (!createRuleParamLists(pRules+i, AF_INET, &ipv4RulesList))
			{
				DbgPrint("nf_setRules() invalid IPv4 rule %d", i);
				status = NF_STATUS_FAIL;
				break;
			}
			if (!createRuleParamLists(pRules+i, AF_INET6, &ipv6RulesList))
			{
				DbgPrint("nf_setRules() invalid IPv6 rule %d", i);
				status = NF_STATUS_FAIL;
				break;
			}
		}

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_MANGLE,
					"-A", g_nfRulesData.driverName_redirect_in,
					"-p", "tcp",
					"-m", "mark",
					"--mark", MARK_IN_S "/" MARK_IN_S,
					"-j", "TPROXY",
					"--on-port", sPort,
					"--on-ip", "127.0.0.1",
					"--tproxy-mark", "0x0/0x0",
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_MANGLE,
					"-A", g_nfRulesData.driverName_redirect_in,
					"-p", "tcp",
					"-m", "mark",
					"--mark", MARK_IN_S "/" MARK_IN_S,
					"-j", "TPROXY",
					"--on-port", sPort,
					"--on-ip", "::ffff:127.0.0.1",
					"--tproxy-mark", "0x0/0x0",
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_MANGLE,
					"-A", FWCTRL_OUTPUT,
					"-j", g_nfRulesData.driverName_redirect_out,
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_MANGLE,
					"-A", FWCTRL_PREROUTING,
					"-j", g_nfRulesData.driverName_redirect_in,
					(char*)NULL));

	#ifndef NO_IPV6_NAT

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_MANGLE,
					"-A", FWCTRL_OUTPUT,
					"-j", g_nfRulesData.driverName_redirect_out,
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_MANGLE,
					"-A", FWCTRL_PREROUTING,
					"-j", g_nfRulesData.driverName_redirect_in,
					(char*)NULL));

	#endif

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_FILTER,
					"-I", FWCTRL_INPUT,
					"-j", g_nfRulesData.driverName_in,
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_FILTER,
					"-I", FWCTRL_OUTPUT,
					"-j", g_nfRulesData.driverName_out,
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_FILTER,
					"-I", FWCTRL_INPUT,
					"-j", g_nfRulesData.driverName_in,
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_FILTER,
					"-I", FWCTRL_OUTPUT,
					"-j", g_nfRulesData.driverName_out,
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_NAT,
					"-I", FWCTRL_OUTPUT,
					"-p", "tcp",
					"-m", "mark",
					"--mark", MARK_IN_S "/" MARK_IN_S,
					"-j", "ACCEPT",
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_NAT,
					"-I", FWCTRL_POSTROUTING,
					"-p", "tcp",
					"-m", "mark",
					"--mark", MARK_IN_S "/" MARK_IN_S,
					"-j", "ACCEPT",
					(char*)NULL));

	#ifndef NO_IPV6_NAT

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_NAT,
					"-I", FWCTRL_OUTPUT,
					"-p", "tcp",
					"-m", "mark",
					"--mark", MARK_IN_S "/" MARK_IN_S,
					"-j", "ACCEPT",
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_NAT,
					"-I", FWCTRL_POSTROUTING,
					"-p", "tcp",
					"-m", "mark",
					"--mark", MARK_IN_S "/" MARK_IN_S,
					"-j", "ACCEPT",
					(char*)NULL));
	#endif

		if (!execRulesList(&ipv4RulesList))
		{
			DbgPrint("nf_setRules() unable to add IPv4 rules");
			status = NF_STATUS_FAIL;
			break;
		}

		if (!execRulesList(&ipv6RulesList))
		{
			DbgPrint("nf_setRules() unable to add IPv6 rules");
		}
		
		break;
	}

	nfrules_paramListListFree(&ipv4RulesList);
	nfrules_paramListListFree(&ipv6RulesList);

	return status;
}

NFAPI_API NF_STATUS NFAPI_NS 
nf_deleteRules()
{
	VLIST_ENTRY ipv4RulesList;
	VLIST_ENTRY ipv6RulesList;
	NF_STATUS status = NF_STATUS_SUCCESS;

	vlist_init(&ipv4RulesList);
	vlist_init(&ipv6RulesList);

	for (;;)
	{
		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_MANGLE,
					"-F", g_nfRulesData.driverName_redirect_out,
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_MANGLE,
					"-F", g_nfRulesData.driverName_redirect_in,
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_MANGLE,
					"-D", FWCTRL_OUTPUT,
					"-j", g_nfRulesData.driverName_redirect_out,
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_MANGLE,
					"-D", FWCTRL_PREROUTING,
					"-j", g_nfRulesData.driverName_redirect_in,
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_MANGLE,
					"-X", g_nfRulesData.driverName_redirect_out,
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_MANGLE,
					"-X", g_nfRulesData.driverName_redirect_in,
					(char*)NULL));

	#ifndef NO_IPV6_NAT

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_MANGLE,
					"-F", g_nfRulesData.driverName_redirect_out,
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_MANGLE,
					"-F", g_nfRulesData.driverName_redirect_in,
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_MANGLE,
					"-D", FWCTRL_OUTPUT,
					"-j", g_nfRulesData.driverName_redirect_out,
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_MANGLE,
					"-D", FWCTRL_PREROUTING,
					"-j", g_nfRulesData.driverName_redirect_in,
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_MANGLE,
					"-X", g_nfRulesData.driverName_redirect_out,
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_MANGLE,
					"-X", g_nfRulesData.driverName_redirect_in,
					(char*)NULL));
	#endif

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_FILTER,
					"-F", g_nfRulesData.driverName_in,
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_FILTER,
					"-D", FWCTRL_INPUT,
					"-j", g_nfRulesData.driverName_in,
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_FILTER,
					"-X", g_nfRulesData.driverName_in,
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_FILTER,
					"-F", g_nfRulesData.driverName_out,
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_FILTER,
					"-D", FWCTRL_OUTPUT,
					"-j", g_nfRulesData.driverName_out,
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_FILTER,
					"-X", g_nfRulesData.driverName_out,
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_FILTER,
					"-F", g_nfRulesData.driverName_in,
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_FILTER,
					"-D", FWCTRL_INPUT,
					"-j", g_nfRulesData.driverName_in,
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_FILTER,
					"-X", g_nfRulesData.driverName_in,
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_FILTER,
					"-F", g_nfRulesData.driverName_out,
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_FILTER,
					"-D", FWCTRL_OUTPUT,
					"-j", g_nfRulesData.driverName_out,
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_FILTER,
					"-X", g_nfRulesData.driverName_out,
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_NAT,
					"-D", FWCTRL_OUTPUT,
					"-p", "tcp",
					"-m", "mark",
					"--mark", MARK_IN_S "/" MARK_IN_S,
					"-j", "ACCEPT",
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV4,
					"-t", FWCTRL_NAT,
					"-D", FWCTRL_POSTROUTING,
					"-p", "tcp",
					"-m", "mark",
					"--mark", MARK_IN_S "/" MARK_IN_S,
					"-j", "ACCEPT",
					(char*)NULL));

	#ifndef NO_IPV6_NAT

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_NAT,
					"-D", FWCTRL_OUTPUT,
					"-p", "tcp",
					"-m", "mark",
					"--mark", MARK_IN_S "/" MARK_IN_S,
					"-j", "ACCEPT",
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamList(
					FWCTRL_PROCESS_IPV6,
					"-t", FWCTRL_NAT,
					"-D", FWCTRL_POSTROUTING,
					"-p", "tcp",
					"-m", "mark",
					"--mark", MARK_IN_S "/" MARK_IN_S,
					"-j", "ACCEPT",
					(char*)NULL));
	#endif

		if (!execRulesList(&ipv4RulesList, false))
		{
			DbgPrint("nf_deleteRules() unable to delete IPv4 rules");
			status = NF_STATUS_FAIL;
			break;
		}

		if (!execRulesList(&ipv6RulesList, false))
		{
			DbgPrint("nf_deleteRules() unable to delete IPv6 rules");
	//		return NF_STATUS_FAIL;
		}

		break;
	}

	nfrules_paramListListFree(&ipv4RulesList);
	nfrules_paramListListFree(&ipv6RulesList);

	return status;
}


NF_STATUS nf_addRouteRules()
{
	VLIST_ENTRY ipv4RulesList;
	VLIST_ENTRY ipv6RulesList;
	NF_STATUS status = NF_STATUS_SUCCESS;

	vlist_init(&ipv4RulesList);
	vlist_init(&ipv6RulesList);

	for (;;)
	{
		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamListGeneric(
					FWCTRL_PROCESS_IP,
					"route", "add",
					"local", "default",
					"dev", "lo",
					"table", MARK_TABLE,
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamListGeneric(
					FWCTRL_PROCESS_IP,
					"rule", "add",
					"fwmark", MARK_IN_S "/" MARK_IN_S,
					"lookup", MARK_TABLE,
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamListGeneric(
					FWCTRL_PROCESS_IP,
					"-6",
					"route", "add",
					"local", "default",
					"dev", "lo",
					"table", MARK_TABLE,
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamListGeneric(
					FWCTRL_PROCESS_IP,
					"-6",
					"rule", "add",
					"fwmark", MARK_IN_S "/" MARK_IN_S,
					"lookup", MARK_TABLE,
					(char*)NULL));

		if (!execRulesList(&ipv4RulesList, false))
		{
			DbgPrint("nf_addRouteRules() unable to add IPv4 rules");
			status = NF_STATUS_FAIL;
			break;
		}

		if (!execRulesList(&ipv6RulesList, false))
		{
			DbgPrint("nf_addRouteRules() unable to add IPv6 rules");
	//		return NF_STATUS_FAIL;
		}
		
		break;
	}

	nfrules_paramListListFree(&ipv4RulesList);
	nfrules_paramListListFree(&ipv6RulesList);

	return status;
}

NF_STATUS nf_deleteRouteRules()
{
	VLIST_ENTRY ipv4RulesList;
	VLIST_ENTRY ipv6RulesList;
	NF_STATUS status = NF_STATUS_SUCCESS;

	vlist_init(&ipv4RulesList);
	vlist_init(&ipv6RulesList);

	for (;;)
	{
		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamListGeneric(
					FWCTRL_PROCESS_IP,
					"route", "del",
					"local", "default",
					"dev", "lo",
					"table", MARK_TABLE,
					(char*)NULL));

		vlist_insertTail(&ipv4RulesList, 
			(VLIST_ENTRY*)createParamListGeneric(
					FWCTRL_PROCESS_IP,
					"rule", "del",
					"fwmark", MARK_IN_S "/" MARK_IN_S,
					"lookup", MARK_TABLE,
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamListGeneric(
					FWCTRL_PROCESS_IP,
					"-6",
					"route", "del",
					"local", "default",
					"dev", "lo",
					"table", MARK_TABLE,
					(char*)NULL));

		vlist_insertTail(&ipv6RulesList, 
			(VLIST_ENTRY*)createParamListGeneric(
					FWCTRL_PROCESS_IP,
					"-6",
					"rule", "del",
					"fwmark", MARK_IN_S "/" MARK_IN_S,
					"lookup", MARK_TABLE,
					(char*)NULL));

		if (!execRulesList(&ipv4RulesList, false))
		{
			DbgPrint("nf_deleteRouteRules() unable to delete IPv4 rules");
			status = NF_STATUS_FAIL;
			break;
		}

		if (!execRulesList(&ipv6RulesList, false))
		{
			DbgPrint("nf_deleteRouteRules() unable to delete IPv6 rules");
	//		return NF_STATUS_FAIL;
		}
		
		break;
	}

	nfrules_paramListListFree(&ipv4RulesList);
	nfrules_paramListListFree(&ipv6RulesList);

	return status;
}

void nfrules_setDriverName(const char * driverName)
{
	_snprintf(g_nfRulesData.driverName, sizeof(g_nfRulesData.driverName), "%s", driverName);
	_snprintf(g_nfRulesData.driverName_in, NF_MAX_NAME, "%s_in", driverName);
	_snprintf(g_nfRulesData.driverName_out, NF_MAX_NAME, "%s_out", driverName);
	_snprintf(g_nfRulesData.driverName_redirect_in, NF_MAX_NAME, "%s_rd_in", driverName);
	_snprintf(g_nfRulesData.driverName_redirect_out, NF_MAX_NAME, "%s_rd_out", driverName);
}