//
// 	NetFilterSDK 
// 	Copyright (C) Vitaly Sidorov
//	All rights reserved.
//
//	This file is a part of the NetFilter SDK.
//	The code and information is provided "as-is" without
//	warranty of any kind, either expressed or implied.
//

#include "vdefs.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/epoll.h>

#include "vsync.h"
#include "vthread.h"
#include "dbglog.h"
#include "nfiocp.h"

using namespace mainNS;

struct NFIOCP_DATA
{
	tOnComplete		onComplete;
	int				poll_fd;
	vsyncEvent_t	threadStopEvent;
	vthread_t		workerThread;
};

static bool g_initialized = false;
static NFIOCP_DATA g_iocpData;

bool mainNS::nfiocp_registerSocket(int s, uint64_t data, int events)
{
	epoll_event ev;

	if (!g_initialized)
		return false;

	ev.events = events;
	ev.data.u64 = data;

	if (epoll_ctl(g_iocpData.poll_fd, EPOLL_CTL_ADD, s, &ev) != 0)
		return false;

	return true;
}

bool mainNS::nfiocp_unRegisterSocket(int s)
{
	epoll_event ev;

	if (!g_initialized)
		return false;

	if (epoll_ctl(g_iocpData.poll_fd, EPOLL_CTL_DEL, s, &ev) != 0)
		return false;

	return true;
}

static VTHREAD_FUNC nfiocp_workerThread(void*)
{
	const int nEvents = 128;
	epoll_event events[nEvents];
	int nEventsRet;
	int i;

	for (;;)
	{
		if (vsyncEvent_isFlagFired(g_iocpData.threadStopEvent, 1))
			break;

		nEventsRet = epoll_wait(g_iocpData.poll_fd, events, nEvents, 500);

		if (nEventsRet == 0)
		{
			g_iocpData.onComplete(0, 0);
			continue;
		}

		if (nEventsRet < 0)
		{
			if (errno == EINTR)
				continue;

			DbgPrint("nfiocp_workerThread epoll_wait error %d", errno);

			break;
		}

		for (i = 0; i < nEventsRet; i++)
		{
			g_iocpData.onComplete(events[i].data.u64, events[i].events);
		}
	}

	return 0;
}

bool mainNS::nfiocp_init(tOnComplete pHandler)
{
	DbgPrint("nfiocp_init");

	if (g_initialized)
		return true;

	if (!pHandler)
	{
		DbgPrint("nfiocp_init failed, no handler");
		return false;
	}

	g_iocpData.onComplete = pHandler;
	g_iocpData.threadStopEvent = NULL;
	g_iocpData.workerThread = 0;
	g_iocpData.poll_fd = -1;

	g_initialized = true;

	for (;;)
	{
		g_iocpData.poll_fd = epoll_create(1);
		if (g_iocpData.poll_fd < 0)
			break;

		g_iocpData.threadStopEvent = vsyncEvent_create();
		if (g_iocpData.threadStopEvent == NULL)
			break;

		g_iocpData.workerThread = vthread_create(nfiocp_workerThread, NULL);
		if (g_iocpData.workerThread == 0)
			break;

		return true;
	}

	DbgPrint("nfiocp_init failed");

	nfiocp_free();
	return false;
}

void mainNS::nfiocp_free()
{
	DbgPrint("nfiocp_free");

	if (!g_initialized)
		return;

	g_initialized = false;

	if (g_iocpData.workerThread != 0)
	{
		vsyncEvent_fire(g_iocpData.threadStopEvent, 1);

		vthread_wait(g_iocpData.workerThread);
	}

	if (g_iocpData.threadStopEvent != NULL)
	{
		vsyncEvent_free(g_iocpData.threadStopEvent);
		g_iocpData.threadStopEvent = NULL;
	}

	if (g_iocpData.poll_fd != -1)
	{
		close(g_iocpData.poll_fd);
		g_iocpData.poll_fd = -1;
	}
}
