//
// 	NetFilterSDK 
// 	Copyright (C) Vitaly Sidorov
//	All rights reserved.
//
//	This file is a part of the NetFilter SDK.
//	The code and information is provided "as-is" without
//	warranty of any kind, either expressed or implied.
//

#include "vdefs.h"
#include <time.h>
#include <string.h>
#include <stdio.h>
#include <stdarg.h>
#include "dbglog.h"
#include "vbuffer.h"
#include "vthread.h"
#include "vsync.h"
#include "vlist.h"

using namespace mainNS;

typedef struct _DBGLOG_DATA
{
	BOOL			enabled;
	BOOL			initialized;
	vbuffer_t		fileName;
	VLIST_ENTRY		logQueue;
	vthread_t		thread;
	vsyncEvent_t	threadEvent;
	vcs_t			cs;
} DBGLOG_DATA;

static DBGLOG_DATA g_logData;

static VTHREAD_FUNC dbglog_workerThread(void*);

BOOL mainNS::dbglog_init(const char* fileName)
{
	memset(&g_logData, 0, sizeof(g_logData));

	for (;;)
	{
		g_logData.cs = vcs_create();
		
		g_logData.fileName = vbuffer_create();
		if (!g_logData.fileName)
			break;

		vbuffer_appendString(g_logData.fileName, fileName);

		vlist_init(&g_logData.logQueue);
		
		g_logData.threadEvent = vsyncEvent_create();
		if (!g_logData.threadEvent)
			break;
		
		g_logData.thread = vthread_create(dbglog_workerThread, NULL);
		
		g_logData.initialized = TRUE;

		if (!g_logData.cs ||
			!g_logData.fileName ||
			!g_logData.threadEvent ||
			!g_logData.thread)
		{
			break;
		}

		return TRUE;
	}

	dbglog_free();

	return FALSE;
}

void mainNS::dbglog_free()
{
	if (!g_logData.initialized)
		return;

	g_logData.initialized = FALSE;

	if (g_logData.threadEvent && g_logData.thread)
	{
		vsyncEvent_fire(g_logData.threadEvent, 2);
		vthread_wait(g_logData.thread);
	}

	if (g_logData.threadEvent)
	{
		vsyncEvent_free(g_logData.threadEvent);
		g_logData.threadEvent = NULL;
	}

	if (g_logData.cs)
	{
		vcs_free(g_logData.cs);
		g_logData.cs = NULL;
	}

	if (g_logData.fileName)
	{
		vbuffer_free(g_logData.fileName);
		g_logData.fileName = NULL;
	}

	{
		vbuffer_t entry;
		while (!vlist_isEmpty(&g_logData.logQueue))
		{
			entry = vbuffer_fromListEntry(vlist_getHead(&g_logData.logQueue));
			vlist_removeHead(&g_logData.logQueue);
			vbuffer_free(entry);
		}
	}
}

void mainNS::dbglog_enable(BOOL enable)
{
	g_logData.enabled = enable;
}

BOOL mainNS::dbglog_enabled()
{
	return g_logData.enabled;
}


static vbuffer_t getTime()
{
	vbuffer_t sTime;
	time_t tNow;
	struct tm tm;
	size_t len;

	sTime = vbuffer_create();
	if (!sTime)
		return NULL;

	tNow = time(NULL);
	localtime_r(&tNow, &tm);

	if (!vbuffer_append(sTime, NULL, 512, FALSE))
		return sTime;

	asctime_r(&tm, vbuffer_buffer(sTime));
	
	len = strnlen(vbuffer_buffer(sTime), vbuffer_getSize(sTime));
	if (len > 0)
		vbuffer_buffer(sTime)[len - 1] = '\0';

	return sTime;
}

void mainNS::dbglog_write(const char* sMessage, ...)
{
	int nSize = 512;
	vbuffer_t bufEntry = NULL;
	vbuffer_t tempBuffer = NULL;
	vbuffer_t timeBuffer = NULL;
	int nResult;
	va_list args;

	if (!g_logData.enabled)
		return;

	for (;;)
	{
		tempBuffer = vbuffer_create();
		if (!tempBuffer)
			break;

		bufEntry = vbuffer_create();
		if (!bufEntry)
			break;

		timeBuffer = getTime();
		if (!timeBuffer)
			break;

		for (;;)
		{
			vbuffer_append(tempBuffer, NULL, nSize, FALSE);

			va_start(args, sMessage);
			nResult = vsnprintf(vbuffer_buffer(tempBuffer),
				vbuffer_getSize(tempBuffer),
				sMessage,
				args);
			va_end(args);

			if (nResult < 0)
			{
				nSize *= 2;
				continue;
			}

			break;
		}

		if (!vbuffer_appendString(bufEntry, vbuffer_buffer(timeBuffer)))
			break;

		if (!vbuffer_appendString(bufEntry, " : "))
			break;

		if (!vbuffer_appendString(bufEntry, vbuffer_buffer(tempBuffer)))
			break;

		if (!vbuffer_appendString(bufEntry, "\r\n"))
			break;

		vcs_lock(g_logData.cs);
		vlist_insertTail(&g_logData.logQueue, vbuffer_getListEntry(bufEntry));
		vcs_unlock(g_logData.cs);

		vsyncEvent_fire(g_logData.threadEvent, 1);

		bufEntry = NULL;
		
		break;
	}

	if (bufEntry)
		vbuffer_free(bufEntry);
	
	if (tempBuffer)
		vbuffer_free(tempBuffer);

	if (timeBuffer)
		vbuffer_free(timeBuffer);
}

static VTHREAD_FUNC dbglog_workerThread(void*p)
{
	FILE* file = NULL;
	vbuffer_t entry;
	int flags = 0;

	file = fopen(vbuffer_buffer(g_logData.fileName), "ab");
	if (file == NULL)
	{
		file = fopen(vbuffer_buffer(g_logData.fileName), "wb");
	}

	if (!file)
	{
		g_logData.enabled = FALSE;
		return 0;
	}

	for (;;)
	{
		if (vsyncEvent_wait(g_logData.threadEvent, &flags, 5))
		{
			if (flags & 2)
			{
				break;
			}
		}

		for (;;)
		{
			vcs_lock(g_logData.cs);
			
			if (vlist_isEmpty(&g_logData.logQueue))
			{
				vcs_unlock(g_logData.cs);
				break;
			}
			
			entry = vbuffer_fromListEntry(vlist_getHead(&g_logData.logQueue));
			
			vlist_removeHead(&g_logData.logQueue);

			vcs_unlock(g_logData.cs);

			fwrite(vbuffer_buffer(entry), 
				sizeof(char), 
				vbuffer_getSize(entry), 
				file);

			fflush(file);

			vbuffer_free(entry);
		}
	}

	fclose(file);

	return 0;
}


